#!/usr/bin/env python3
"""
生成Chrome插件图标
需要安装: pip install Pillow
"""

from PIL import Image, ImageDraw, ImageFont
import os

def create_icon(size):
    """创建指定尺寸的图标"""
    # 创建图像
    img = Image.new('RGBA', (size, size), (0, 0, 0, 0))
    draw = ImageDraw.Draw(img)
    
    # 绘制圆形背景（渐变色的简化版，使用纯色）
    margin = size // 10
    circle_bbox = [margin, margin, size - margin, size - margin]
    draw.ellipse(circle_bbox, fill='#667eea')
    
    # 绘制灯泡轮廓（简化版）
    center_x = size // 2
    center_y = size // 2
    
    # 灯泡主体
    bulb_size = size // 3
    bulb_top = center_y - bulb_size
    bulb_bottom = center_y + bulb_size // 2
    
    # 绘制灯泡形状（简化为圆形）
    bulb_bbox = [
        center_x - bulb_size // 2,
        bulb_top,
        center_x + bulb_size // 2,
        bulb_bottom
    ]
    draw.ellipse(bulb_bbox, fill='white', outline='white', width=2)
    
    # 绘制灯泡底座（矩形）
    base_height = size // 16
    base_bbox = [
        center_x - bulb_size // 3,
        bulb_bottom,
        center_x + bulb_size // 3,
        bulb_bottom + base_height
    ]
    draw.rectangle(base_bbox, fill='white')
    
    # 添加"AI"文字（如果尺寸足够大）
    if size >= 48:
        try:
            # 尝试使用系统字体
            font_size = size // 5
            try:
                font = ImageFont.truetype("/System/Library/Fonts/Helvetica.ttc", font_size)
            except:
                font = ImageFont.load_default()
            
            text = "AI"
            # 获取文字尺寸
            bbox = draw.textbbox((0, 0), text, font=font)
            text_width = bbox[2] - bbox[0]
            text_height = bbox[3] - bbox[1]
            
            # 在底部绘制文字
            text_x = center_x - text_width // 2
            text_y = size - size // 5
            draw.text((text_x, text_y), text, fill='white', font=font)
        except Exception as e:
            print(f"添加文字失败: {e}")
    
    return img

def main():
    """生成所有尺寸的图标"""
    sizes = [16, 32, 48, 128]
    
    # 获取脚本所在目录
    script_dir = os.path.dirname(os.path.abspath(__file__))
    
    print("开始生成图标...")
    
    for size in sizes:
        filename = f"icon{size}.png"
        filepath = os.path.join(script_dir, filename)
        
        try:
            img = create_icon(size)
            img.save(filepath, 'PNG')
            print(f"✓ 已生成 {filename} ({size}x{size})")
        except Exception as e:
            print(f"✗ 生成 {filename} 失败: {e}")
    
    print("\n图标生成完成！")
    print("请检查 assets/icons/ 目录下的图标文件。")

if __name__ == "__main__":
    try:
        main()
    except ImportError:
        print("错误：需要安装 Pillow 库")
        print("请运行: pip install Pillow")
    except Exception as e:
        print(f"发生错误: {e}")

